<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\QrCode;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;

class AdminController extends Controller
{
    public function dashboard()
    {
        return view('admin.dashboard');
    }

    public function createUser()
    {
        return view('admin.create-user');
    }

    public function storeUser(Request $request)
    {
        // Validate the input data
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8|confirmed',
            'role' => 'required|in:99,1',
        ]);

        // Create the new user
        User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'role' => $validated['role'],
        ]);

        return redirect()->route('admin.create-user')->with('success', 'User created successfully!');
    }

    public function qrGenerator(Request $request)
    {
        if ($request->isMethod('post')) {
            $labelInput = $request->input('label');
            $day = $request->input('day');
            $quantity = $request->input('quantity', 1);
            $generatedQrs = [];
        
            $labelPrefix = $labelInput . ' Complimentary Pass';
        
            // Get the highest existing number with the same label prefix
            $lastLabel = \App\Models\QrCode::where('label', 'like', "$labelPrefix%")
                ->orderByDesc(DB::raw("CAST(SUBSTRING_INDEX(label, ' ', -1) AS UNSIGNED)"))
                ->value('label');
        
            if ($lastLabel) {
                preg_match('/(\d+)$/', $lastLabel, $matches);
                $nextNumber = isset($matches[1]) ? intval($matches[1]) + 1 : 1;
            } else {
                $nextNumber = 1;
            }
        
            for ($i = 0; $i < $quantity; $i++) {
                $code = uniqid();
                $link = url("/validate-qr?code={$code}&day={$day}");
                $filename = "qr_{$code}.png";
        
                $fullLabel = $labelPrefix . ' ' . ($nextNumber + $i);
        
                QrCode::create([
                    'user_id' => auth()->id(),
                    'label' => $fullLabel,
                    'day' => $day,
                    'code' => $code,
                ]);
        
                $qr = \SimpleSoftwareIO\QrCode\Facades\QrCode::format('png')->size(350)->generate($link);
        
                Storage::put("public/qrcodes/{$filename}", $qr);
        
                $generatedQrs[] = [
                    'svg' => $filename,
                    'link' => $link,
                    'label' => $fullLabel,
                ];
            }
        
            return back()->with([
                'generatedQrs' => $generatedQrs,
                'success' => 'Batch QR Codes generated successfully!'
            ]);
        }

        return view('admin.qr-generator');
    }

    public function qrList()
    {
        $qrs = QrCode::latest()->get(); // Paginate to avoid loading too many at once
        return view('admin.qr-list', compact('qrs'));
    }
}

