<?php
/**
 * Deep Malware Scanner and Cleaner for WordPress-based Sites
 * ⚠️ Always BACK UP your files before running this script!
 */

set_time_limit(0);
ini_set('memory_limit', '1024M');

// $rootDir = __DIR__.'/staging/cbzncorp_revamp';

$rootDir = __DIR__;
$logFile = $rootDir . '/deep_malware_log.txt';

$replaceInfected = true;   // Set to false to only log
$cleanIndex = true;        // Clean index.php by replacing
$cleanFunctions = true;    // Attempt to clean functions.php (safely)

$suspiciousPatterns = [
    '/iplogger\.co/i',
    '/surl\.li/i',
    '/shorturl\.at/i',
    '/confirmation_required/i',
    '/desktop_links/i',
    '/mobile_links/i',
    '/blocked_bots/i',
    '/log_visit/i',
    '/base64_decode\s*\(/i',
    '/gzinflate\s*\(/i',
    '/shell_exec\s*\(/i',
    '/eval\s*\(/i',
    '/preg_replace\s*\(\s*[\'"]\/.*\/e[\'"]\s*,/i',
    '/system\s*\(/i',
    '/passthru\s*\(/i',
    '/exec\s*\(/i',
    '/[\'"]wp-config-sample\.php[\'"]/i'
];

$infectedFiles = [];

$defaultIndex = "<?php\n/**\n * Front to the WordPress application. This file doesn't do anything, but loads\n * wp-blog-header.php which does and tells WordPress to load the theme.\n */\n\ndefine('WP_USE_THEMES', true);\n\nrequire __DIR__ . '/wp-blog-header.php';\n";

// ✅ Improved safe cleaner for functions.php
function cleanInjectedHeader($content) {
    $lines = explode("\n", $content);
    $cleanLines = [];
    $injected = false;
    $foundPhpStart = false;

    foreach ($lines as $line) {
        $trimmed = trim($line);

        if (!$foundPhpStart && strpos($trimmed, '<?php') !== false) {
            $foundPhpStart = true;
            $cleanLines[] = $trimmed;
            continue;
        }

        if ($foundPhpStart) {
            if (preg_match('/(base64_decode|gzinflate|eval|shell_exec|system|passthru|exec|iplogger|surl\.li|shorturl\.at)/i', $trimmed)) {
                $injected = true;
                continue; // Remove malicious line
            }

            $cleanLines[] = $line;
        } else {
            $cleanLines[] = $line; // Keep comments before PHP tag
        }
    }

    return $injected ? implode("\n", $cleanLines) : $content;
}

function scanDirectory($dir) {
    global $suspiciousPatterns, $infectedFiles, $replaceInfected, $defaultIndex, $cleanIndex, $cleanFunctions;

    $items = scandir($dir);
    foreach ($items as $item) {
        if ($item === '.' || $item === '..') continue;

        $path = $dir . DIRECTORY_SEPARATOR . $item;

        if (is_dir($path)) {
            scanDirectory($path);
        } elseif (pathinfo($path, PATHINFO_EXTENSION) === 'php') {
            $contents = @file_get_contents($path);
            if ($contents === false) continue;

            foreach ($suspiciousPatterns as $pattern) {
                if (preg_match($pattern, $contents)) {
                    $infectedFiles[] = $path;

                    if ($replaceInfected) {
                        if ($cleanIndex && basename($path) === 'index.php') {
                            @file_put_contents($path, $defaultIndex);
                        } elseif ($cleanFunctions && basename($path) === 'functions.php') {
                            $cleaned = cleanInjectedHeader($contents);
                            @file_put_contents($path, $cleaned);
                        }
                    }

                    break; // Stop after first match
                }
            }
        }
    }
}

echo "🔍 Scanning all PHP files under: $rootDir\n";
scanDirectory($rootDir);

file_put_contents($logFile, implode(PHP_EOL, $infectedFiles));

echo "✅ Scan complete.\n";
echo "🦠 Infected files found: " . count($infectedFiles) . "\n";
echo "📄 Log saved at: $logFile\n";

if ($replaceInfected) {
    echo "🧹 index.php and functions.php were cleaned where applicable.\n";
}
?>