<?php

namespace NewfoldLabs\WP\Module\Hosting\MalwareCheck;

use NewfoldLabs\WP\Module\Installer\Services\PluginInstaller;
use NewfoldLabs\WP\Module\Hosting\Helpers\APIHelper;
use NewfoldLabs\WP\Module\Hosting\HostingPanel\HostingPanel;

/**
 * Handles malware check-related functionality.
 */
class MalwareCheck {

	/**
	 * Dependency container instance.
	 *
	 * @var mixed
	 */
	protected $container;

	/**
	 * MalwareCheck constructor.
	 *
	 * @param mixed $container The dependency container instance.
	 */
	public function __construct( $container ) {
		$this->container = $container;
	}

	/**
	 * Retrieves malware scan config and cached scan status (if any).
	 *
	 * @return array
	 */
	public function get_data() {
		$plugin_basename = $this->container->plugin()->brand;

		// Get existing panel transient
		$transient_key = HostingPanel::$transient_key;
		$cached_data   = get_transient( $transient_key );
		$scan_status   = 'unknown';

		if ( ! empty( $cached_data ) && isset( $cached_data['malware-check']['scan_status'] ) ) {
			$scan_status = $cached_data['malware-check']['scan_status'];
		}

		return array(
			'registration_nonce' => wp_create_nonce( 'jetpack-registration-nonce' ),
			'install_token'      => PluginInstaller::rest_get_plugin_install_hash(),
			'plugin'             => 'jetpack-protect',
			'api'                => array(
				'jetpackRegister' => APIHelper::get_rest_api_url( '/jetpack/v4/connection/register' ),
				'scanStatus'      => APIHelper::get_rest_api_url(
					'/jetpack-protect/v1/status',
					array(
						'hard_refresh' => 'true',
						'_locale'      => 'user',
					)
				),
				'pluginInstall'   => APIHelper::get_rest_api_url( '/newfold-installer/v1/plugins/install' ),
			),
			'urls'               => array(
				'scanPage'    => admin_url( 'admin.php?page=jetpack-protect#/scan' ),
				'supportPage' => admin_url( "admin.php?page={$plugin_basename}#/help" ),
			),
			'scan_status'        => $scan_status,
		);
	}

	/**
	 * Updates the scan status inside the main hosting panel transient if it exists.
	 *
	 * @param array $data The array containing scan status info. Must include 'scan_status'.
	 *
	 * @return bool True if the transient was updated, false otherwise.
	 */
	public static function update_scan_status( $data ) {
		if ( empty( $data['scan_status'] ) ) {
			return false;
		}

		$status        = sanitize_text_field( $data['scan_status'] );
		$transient_key = HostingPanel::$transient_key;
		$cached_data   = get_transient( $transient_key );

		if ( ! empty( $cached_data ) && is_array( $cached_data ) ) {
			$cached_data['malware-check']['scan_status'] = $status;
			set_transient( $transient_key, $cached_data, DAY_IN_SECONDS );
			return true;
		}

		return false;
	}
}
